/*
 * linux/arch/arm/mach-uniphier/irq.c
 *
 * Copyright (C) 2011 Panasonic Corporation
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */
#include <linux/module.h>
#include <linux/irq.h>
#include <linux/interrupt.h>

#include <asm/io.h>
#include <asm/hardware/gic.h>
#include <mach/hardware.h>


static void mn_cpupic_noop(unsigned int irq)
{
}

static struct irq_chip mn_cpu_pic_empty = {
	.name		= "cpu_empty",
	.disable	= mn_cpupic_noop,
	.enable		= mn_cpupic_noop,
	.ack		= mn_cpupic_noop,
	.mask		= mn_cpupic_noop,
	.mask_ack	= mn_cpupic_noop,
	.unmask		= mn_cpupic_noop,
};


#define UNIPHIER_GIC_PRI_SHIFT	4

extern spinlock_t irq_controller_lock;

static inline u32 __get_icr_u32(u32 reg)
{
	unsigned long flags;
	u32 tmp32;

	spin_lock_irqsave(&irq_controller_lock, flags);
	tmp32 = inl(reg);
	spin_unlock_irqrestore(&irq_controller_lock, flags);
	return tmp32;
}


static inline void __set_icr_u32(u32 val, u32 reg)
{
	unsigned long flags;

	spin_lock_irqsave(&irq_controller_lock, flags);
	outl(val, reg);
	spin_unlock_irqrestore(&irq_controller_lock, flags);
}


/*
 * change the level at which an IRQ executes
 * - must not be called whilst interrupts are being processed!
 */
void mn_intc_set_level(unsigned int irq, unsigned int pri)
{
	unsigned long flags;
	int shift;
	u32 mask, new_pri, tmp32, reg;

	if (in_interrupt())
		BUG();

	shift = (irq % 4) * 8;
	mask = 0xff << shift;
	new_pri = ((pri << UNIPHIER_GIC_PRI_SHIFT) << shift) & mask;
	reg = UNIPHIER_GIC_DIST_START + GIC_DIST_PRI + (irq / 4) * 4;

	spin_lock_irqsave(&irq_controller_lock, flags);
	tmp32 = inl(reg);
	tmp32 = (tmp32 & ~mask) | new_pri;
	outl(tmp32, reg);
	spin_unlock_irqrestore(&irq_controller_lock, flags);
}
EXPORT_SYMBOL(mn_intc_set_level);


void mn_intc_clear(unsigned int irq)
{
	unsigned long flags;
	u32 bit = 1 << (irq % 32);
	u32 status;

	spin_lock_irqsave(&irq_controller_lock, flags);
	status = inl(UNIPHIER_GIC_DIST_START + GIC_DIST_ACTIVE_BIT + (irq / 32) * 4);
	if (status & bit) {
		outl(irq, UNIPHIER_GIC_CPU_START + GIC_CPU_EOI);
	}
	else {
		outl(bit, UNIPHIER_GIC_DIST_START + GIC_DIST_PENDING_CLEAR + (irq / 32) * 4);
	}
	spin_unlock_irqrestore(&irq_controller_lock, flags);
}
EXPORT_SYMBOL(mn_intc_clear);


void mn_intc_set(unsigned int irq)
{
	u32 bit = 1 << (irq % 32);
	u32 reg = UNIPHIER_GIC_DIST_START + GIC_DIST_PENDING_SET + (irq / 32) * 4;

	__set_icr_u32(bit, reg);
}
EXPORT_SYMBOL(mn_intc_set);


void mn_intc_enable(unsigned int irq)
{
	u32 bit = 1 << (irq % 32);
	u32 reg = UNIPHIER_GIC_DIST_START + GIC_DIST_ENABLE_SET + (irq / 32) * 4;

	__set_icr_u32(bit, reg);
}
EXPORT_SYMBOL(mn_intc_enable);


void mn_intc_disable(unsigned int irq)
{
	u32 bit = 1 << (irq % 32);
	u32 reg = UNIPHIER_GIC_DIST_START + GIC_DIST_ENABLE_CLEAR + (irq / 32) * 4;

	__set_icr_u32(bit, reg);
}
EXPORT_SYMBOL(mn_intc_disable);


extern void gic_set_irq_chip(unsigned int gic_nr, unsigned int irq);

void mn_set_lateack_irq_type(int irq)
{
	gic_set_irq_chip(0, irq);
}
EXPORT_SYMBOL(mn_set_lateack_irq_type);


void mn_set_empty_irq_type(int irq)
{
	set_irq_chip(irq, &mn_cpu_pic_empty);
}
EXPORT_SYMBOL(mn_set_empty_irq_type);


unsigned int mn_intc_get_level(unsigned int irq)
{
	int shift;
	u32 mask, tmp32, reg;

	shift = (irq % 4) * 8;
	mask = 0xff << shift;
	reg = UNIPHIER_GIC_DIST_START + GIC_DIST_PRI + (irq / 4) * 4;

	tmp32 = __get_icr_u32(reg);

	return (unsigned int)(((tmp32 & mask) >> shift) >> UNIPHIER_GIC_PRI_SHIFT);
}


unsigned int mn_intc_get_running_priority(void)
{
	unsigned long flags;
	unsigned int pri;
	u32 reg = UNIPHIER_GIC_CPU_START + GIC_CPU_RUNNINGPRI;

	local_irq_save(flags);
	pri = inl(reg);
	local_irq_restore(flags);
	return pri;
}

#define UNIPHIER_INTC_MAX_PREEMPT_COUNT	16

static unsigned int cpu0_mask_priority[UNIPHIER_INTC_MAX_PREEMPT_COUNT];
static unsigned int cpu1_mask_priority[UNIPHIER_INTC_MAX_PREEMPT_COUNT];

unsigned int *uniphier_intc_mask_stack[] = {
	cpu0_mask_priority,
	cpu1_mask_priority
};

unsigned int __mn_intc_get_mask_priority(void)
{
	u32 reg = UNIPHIER_GIC_CPU_START + GIC_CPU_PRIMASK;
	return inl(reg) >> UNIPHIER_GIC_PRI_SHIFT;
}


unsigned int mn_intc_get_mask_priority(void)
{
	unsigned int pri;
	unsigned long flags;

	local_irq_save(flags);
	pri = __mn_intc_get_mask_priority();
	local_irq_restore(flags);
	return pri;
}


void __mn_intc_set_mask_priority(unsigned int pri)
{
	u32 reg = UNIPHIER_GIC_CPU_START + GIC_CPU_PRIMASK;
	outl(pri << UNIPHIER_GIC_PRI_SHIFT, reg);
}


void mn_intc_set_mask_priority(unsigned int pri)
{
	unsigned long flags;
	local_irq_save(flags);
	__mn_intc_set_mask_priority(pri);
	local_irq_restore(flags);
}

void __mn_intc_pop_mask_priority(void)
{
	unsigned int cpu, *p;

#ifdef CONFIG_SMP
	cpu = hard_smp_processor_id();
#else /* CONFIG_SMP */
	cpu = 0;
#endif /* CONFIG_SMP */

	p = uniphier_intc_mask_stack[cpu];
	__mn_intc_set_mask_priority(*(--p) >> UNIPHIER_GIC_PRI_SHIFT);
	uniphier_intc_mask_stack[cpu] = p;
}
